# バッチ設計書 11-npm run test-e2e

## 概要

本ドキュメントは、three.jsプロジェクトにおけるE2E（End-to-End）テストバッチ `npm run test-e2e` の設計仕様を定義する。

### 本バッチの処理概要

Puppeteerを使用してexamplesディレクトリ配下のHTMLサンプルを自動レンダリングし、既存の参照スクリーンショットとピクセル単位で比較することで、three.jsライブラリの描画結果が期待通りであることを検証するE2Eテストバッチである。

**業務上の目的・背景**：three.jsはWebGL/WebGPUを使用した3Dグラフィックスライブラリであり、コードの変更がビジュアル出力に影響を与える可能性がある。このバッチは、リグレッション（機能後退）を早期に検出し、ライブラリの品質を維持するために必要不可欠である。手動での目視確認は非効率かつ見落としが発生しやすいため、自動化されたスクリーンショット比較によって客観的かつ再現性のある品質検証を実現する。

**バッチの実行タイミング**：PRトリガーで自動実行（GitHub Actions CI）、または開発者によるローカル環境での手動実行。

**主要な処理内容**：
1. ローカルHTTPサーバーの起動（ポート1234）
2. Puppeteerによるヘッドレスブラウザの起動（SwiftShader使用）
3. examplesディレクトリのHTMLファイル一覧取得
4. 各exampleのページ読み込みとレンダリング待機
5. スクリーンショット撮影と参照画像との比較
6. 差分がしきい値を超えた場合のエラー報告

**前後の処理との関連**：`npm run build-module`によるビルド完了後に実行。CI環境では`ci.yml`の`e2e`ジョブとして並列実行される。失敗時は`npm run make-screenshot`で参照画像の再生成が必要。

**影響範囲**：examples/ディレクトリ配下の全HTMLサンプル、examples/screenshots/配下の参照画像、test/e2e/output-screenshots/への差分画像出力。

## バッチ種別

E2Eテスト / ビジュアルリグレッションテスト

## 実行スケジュール

| 項目 | 内容 |
|-----|------|
| 実行頻度 | 随時（PRごと、手動実行） |
| 実行時刻 | 不定 |
| 実行曜日 | 不定 |
| 実行日 | 不定 |
| トリガー | PRイベント / 手動 |

## 実行条件

### 前提条件

| 条件 | 説明 |
|-----|------|
| Node.js環境 | Node.js 24以上がインストールされていること |
| 依存パッケージ | `npm ci`による依存関係のインストール完了 |
| ビルド完了 | `npm run build-module`によるthree.jsビルドファイルの生成完了 |
| 参照スクリーンショット | examples/screenshots/に参照画像が存在すること |

### 実行可否判定

- build/ディレクトリにthree.core.js、three.module.js、three.webgpu.jsが存在すること
- examples/ディレクトリにHTMLファイルが存在すること

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | デフォルト値 | 説明 |
|-------------|-----|-----|-------------|------|
| --webgpu | フラグ | No | false | WebGPU関連のexamplesのみを対象とする |
| --make | フラグ | No | false | テストではなくスクリーンショット生成モードで実行 |
| [examples] | 文字列リスト | No | 全examples | 特定のexampleのみをテスト対象とする |

### 入力データソース

| データソース | 形式 | 説明 |
|-------------|------|------|
| examples/*.html | HTMLファイル | テスト対象のサンプルページ |
| examples/screenshots/*.jpg | JPEGファイル | 参照スクリーンショット画像 |
| build/three.*.js | JavaScriptファイル | ビルド済みthree.jsライブラリ |
| test/e2e/deterministic-injection.js | JavaScriptファイル | 決定論的テスト用のスクリプト |
| test/e2e/clean-page.js | JavaScriptファイル | ページクリーンアップ用スクリプト |

## 出力仕様

### 出力データ

| 出力先 | 形式 | 説明 |
|-------|------|------|
| test/e2e/output-screenshots/ | ディレクトリ | 差分画像の出力先 |
| stdout | テキスト | テスト結果のコンソール出力 |

### 出力ファイル仕様

| 項目 | 内容 |
|-----|------|
| ファイル名 | {example名}-actual.jpg, {example名}-expected.jpg, {example名}-diff.jpg |
| 出力先 | test/e2e/output-screenshots/ |
| 文字コード | バイナリ（JPEG） |
| 区切り文字 | N/A |

## 処理フロー

### 処理シーケンス

```
1. サーバー起動
   └─ utils/server.jsを使用してポート1234でHTTPサーバーを起動

2. ブラウザ起動
   └─ Puppeteerでヘッドレスブラウザを起動（SwiftShaderを使用）

3. exampleファイル一覧取得
   └─ examples/ディレクトリからHTMLファイルを列挙、例外リストを除外

4. CI並列化対応
   └─ CI環境変数に基づいてファイルを5分割し担当範囲を決定

5. 各exampleに対するテスト実行
   └─ ページ読み込み → レンダリング待機 → スクリーンショット撮影 → 比較

6. 結果出力
   └─ 成功/失敗のサマリーをコンソール出力

7. クリーンアップ
   └─ ブラウザとサーバーを終了
```

### フローチャート

```mermaid
flowchart TD
    A[バッチ開始] --> B[HTTPサーバー起動]
    B --> C[Puppeteerブラウザ起動]
    C --> D[examplesファイル一覧取得]
    D --> E{CI環境?}
    E -->|Yes| F[ファイルを5分割して担当範囲決定]
    E -->|No| G[全ファイルを対象]
    F --> H[各exampleをループ]
    G --> H
    H --> I[ページ読み込み]
    I --> J[決定論的スクリプト注入]
    J --> K[レンダリング待機]
    K --> L[スクリーンショット撮影]
    L --> M{参照画像存在?}
    M -->|Yes| N[ピクセル比較]
    M -->|No| O[エラー記録]
    N --> P{差分 < 0.3%?}
    P -->|Yes| Q[成功ログ出力]
    P -->|No| R[差分画像保存]
    R --> O
    Q --> S{次のexample?}
    O --> S
    S -->|Yes| H
    S -->|No| T[結果サマリー出力]
    T --> U[ブラウザ・サーバー終了]
    U --> V[バッチ終了]
```

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（データベースを使用しない）

### テーブル別操作詳細

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | ネットワークタイムアウト | ページ読み込みが5分を超過 | 例外リストに追加、または処理をスキップ |
| - | レンダータイムアウト | レンダリングが5秒を超過 | 例外リストに追加検討 |
| - | 参照画像なし | screenshots/に対応画像がない | make-screenshotで生成 |
| - | 差分超過 | ピクセル差分が0.3%を超過 | コード確認または参照画像再生成 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 2回（numAttempts = 2） |
| リトライ間隔 | 即時 |
| リトライ対象エラー | 全エラー |

### 障害時対応

1. 失敗したexample一覧がコンソールに出力される
2. `npm run make-screenshot [失敗したexample名]`で参照画像を再生成
3. 再生成でも解決しない場合は例外リスト（exceptionList）への追加を検討

## トランザクション仕様

| 項目 | 内容 |
|-----|------|
| トランザクション範囲 | 該当なし |
| コミットタイミング | 該当なし |
| ロールバック条件 | 該当なし |

## パフォーマンス要件

| 項目 | 内容 |
|-----|------|
| 想定処理件数 | 約300-400 examples |
| 目標処理時間 | 30分以内（CI環境、5並列時） |
| メモリ使用量上限 | ブラウザ依存 |

## 排他制御

- 同一環境での同時実行は非推奨（ポート競合の可能性）
- CI環境では5スレッド並列実行されるが、各スレッドは異なるCI番号で異なるexampleを担当

## ログ出力

| ログ種別 | 出力タイミング | 出力内容 |
|---------|--------------|---------|
| 開始ログ | サーバー起動時 | （暗黙的） |
| 進捗ログ | 各example処理完了時 | "Diff X.X% in file: {example名}" |
| 終了ログ | バッチ終了時 | "TEST PASSED! X screenshots rendered correctly." |
| エラーログ | 差分超過時 | "Diff wrong in X.X% of pixels in file: {example名}" |
| 警告ログ | 処理継続可能な問題発生時 | 黄色でコンソール出力 |

## 監視・アラート

| 監視項目 | 閾値 | アラート先 |
|---------|-----|----------|
| 処理時間 | 30分（timeout-minutes: 30） | GitHub Actions |
| 失敗example数 | 1件以上 | GitHub Actions PR check |

## 備考

- SwiftShaderを使用することでGPUなし環境でもWebGLレンダリングが可能
- 決定論的テストのため、Math.random、Date.now、performance.nowを固定値に置換
- 約60件のexampleが例外リスト（exceptionList）に登録されており、テストから除外される
- WebGPU関連の多くのexampleは現在例外リストに含まれている
